/*
 *  SVD.c
 *  Matrix Plugin
 *
 *  Created by Robert Delaney on 12/12/09.
 *  Copyright 2009 Bob Delaney's Science Software. All rights reserved.
 *
 */

#include "SVD.h"
#include "diags.h"

// V will diagonalize M' * M
// give this m , n, V (n x n), and M (m x n), where m >= n
// it returns U (m x n) and the vector D (n x 1) of ordered singular values
// for the SVD (with D now a square diagonal matrix (n x n) with the singular values along its diagonal:
// M = U * D * V'   where V' is the transpose of V
// All matrices should be dynamically allocated by the calling program.
void svd(double **U, double *D, double **V, double **M, long m, long n, double eps)
{
	long		i, j, k, jSave;
	double		**A, **U1, *D1, temp;
	
	A = (double **)malloc(n*sizeof(double *));
	for (i=0;i<n;i++)
		A[i] = (double *)malloc(n*sizeof(double));
	
	U1 = (double **)malloc(m*sizeof(double *));
	for (i=0;i<m;i++)
		U1[i] = (double *)malloc(n*sizeof(double));
	
	D1 = (double *)malloc(n*sizeof(double));
	
	// A(n x n) = M' * M  A_ik = Sum(j=0 to m) M_j i * M_j k   i,k = 1, 2, ... , n
	for(i=0;i<n;i++)
		for(k=0;k<n;k++)
		{
			A[i][k] = 0;
			for(j=0;j<m;j++)
				A[i][k] = A[i][k] + M[j][i] * M[j][k];
		}
	
	// diagonalize A with V to hold eigenvectors
	diags(A, V, n, eps);
	
	// form U = M * V
	// U_i k = Sum(j=1 to n) M_i j * V_j k    i = 1 , 2, ... , m   k = 1, 2, ... , n
	
	for(i=0;i<m;i++)
		for(k=0;k<n;k++)
		{
			U1[i][k] = 0;
			for(j=0;j<n;j++)
				U1[i][k] = U1[i][k] + M[i][j] * V[j][k];
		}
	
	// the columns of U are orthogonal vectors, their norms are the singular values
	
	for(j=0;j<n;j++)
	{
		D1[j] = 0;
		for(i=0;i<m;i++)
			D1[j] = D1[j] + U[i][j]*U[i][j];
		D1[j] = sqrt(D1[j]);
	}
	
	// divide each column by its norm unless the norm is zero
	for(j=0;j<n;j++)
	{
		for(i=0;i<m;i++)
		{
			if(D1[j])
				U1[i][j] = U1[i][j] / D1[j];
		}
	}
	
	// now reorder
	k = 0;  // current column index for U and current row index for D
	jSave = 0; // to avoid warning
	while(true)
	{
		temp = -1;
		for(j=0;j<n;j++)
		{
			if(D1[j]>temp)
			{
				temp = D1[j];
				jSave = j; // saves this j
			}
			
			if(temp==-1)
				break;
			
			D[k] = D1[jSave];
			D1[jSave] = -1;
			
			for(i=0;i<m;i++)
				U[i][jSave] = U1[i][k];
			
			k++;
		}
	}
	
	for (i=0;i<n;i++)
		free(A[i]);
	free(A);
		
	for (i=0;i<m;i++)
		free(U1[i]);
	free(U1);
	
	free(D1);
	
}/* svd */

